using UnityEngine;
using HutongGames.PlayMaker;

namespace HutongGames.PlayMaker.Actions
{
    [ActionCategory("LV/Manager")]
    [HutongGames.PlayMaker.Tooltip("経験値を計算し、レベルアップ判定と必要経験値の上昇を行います。")]
    public class LvExpManager : FsmStateAction
    {
        [ActionSection("Inputs")]
        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("獲得した生の経験値（倒したピンの数など）")]
        public FsmFloat gainedExp;

        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("経験値取得率（1.0 = 100%）。強化パラメーター用。")]
        public FsmFloat expGainRate;

        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("次のレベルに必要な経験値の上昇率（例：1.2 = 20%増）")]
        public FsmFloat growthRate;

        [ActionSection("Persistent Variables")]
        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("現在の累計経験値（バーの表示用など）")]
        public FsmFloat currentExp;

        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("現在のレベルアップに必要なしきい値")]
        public FsmFloat targetExp;

        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("ストックされているレベルアップポイント（未消費の強化回数）")]
        public FsmInt levelUpPoints;

        [ActionSection("Events")]
        [HutongGames.PlayMaker.Tooltip("レベルアップが発生した際に送るイベント")]
        public FsmEvent levelUpEvent;

        public override void Reset()
        {
            gainedExp = null;
            expGainRate = 1.0f;
            growthRate = 1.2f;
            currentExp = null;
            targetExp = 100f;
            levelUpPoints = null;
            levelUpEvent = null;
        }

        public override void OnEnter()
        {
            DoExpCalculation();
            Finish();
        }

        void DoExpCalculation()
        {
            // 変数が未設定、または獲得経験値が0以下の場合は処理しない
            if (gainedExp == null || currentExp == null || targetExp == null || levelUpPoints == null) return;
            if (gainedExp.Value <= 0) return;

            // 1. 取得率を適用して加算
            float multiplier = (expGainRate != null) ? expGainRate.Value : 1.0f;
            currentExp.Value += gainedExp.Value * multiplier;

            bool leveledUp = false;

            // 2. レベルアップ判定（複数レベルアップにも対応）
            // targetExpが0以下だと無限ループになるためガード
            while (targetExp.Value > 0 && currentExp.Value >= targetExp.Value)
            {
                currentExp.Value -= targetExp.Value;
                levelUpPoints.Value += 1;

                // 次の必要経験値を上昇させる
                float growth = (growthRate != null) ? growthRate.Value : 1.0f;
                targetExp.Value *= growth;

                leveledUp = true;
            }

            // 3. レベルアップイベントの送信
            if (leveledUp && levelUpEvent != null)
            {
                Fsm.Event(levelUpEvent);
            }
        }
    }
}