using UnityEngine;
using HutongGames.PlayMaker;

namespace HutongGames.PlayMaker.Actions
{
    [ActionCategory("LV/Math")]
    [HutongGames.PlayMaker.Tooltip("Rounds a Float to a specified number of decimal places using the selected method.")]
    public class LvFloatRoundDecimal : FsmStateAction
    {
        public enum RoundingMethod
        {
            Round,     // ľܓ
            Ceil,      // ؂グ
            Floor,     // ؂艺
            Truncate   // ؂̂āi0j
        }

        [RequiredField]
        [HutongGames.PlayMaker.Tooltip("The float value to round.")]
        public FsmFloat floatVariable;

        [HutongGames.PlayMaker.Tooltip("Number of decimal places to round to (e.g., 1 for 0.1).")]
        public FsmInt decimalPlaces;

        [HutongGames.PlayMaker.Tooltip("The rounding method to use.")]
        public RoundingMethod roundingMethod = RoundingMethod.Round;

        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("Store the rounded result here.")]
        public FsmFloat result;

        [HutongGames.PlayMaker.Tooltip("Repeat every frame.")]
        public bool everyFrame;

        public override void Reset()
        {
            floatVariable = null;
            decimalPlaces = 1;
            roundingMethod = RoundingMethod.Round;
            result = null;
            everyFrame = false;
        }

        public override void OnEnter()
        {
            DoRounding();
            if (!everyFrame)
            {
                Finish();
            }
        }

        public override void OnUpdate()
        {
            DoRounding();
        }

        private void DoRounding()
        {
            float multiplier = Mathf.Pow(10, decimalPlaces.Value);
            float scaled = floatVariable.Value * multiplier;
            float rounded;

            switch (roundingMethod)
            {
                case RoundingMethod.Round:
                    rounded = Mathf.Round(scaled);
                    break;
                case RoundingMethod.Ceil:
                    rounded = Mathf.Ceil(scaled);
                    break;
                case RoundingMethod.Floor:
                    rounded = Mathf.Floor(scaled);
                    break;
                case RoundingMethod.Truncate:
                    rounded = (float)System.Math.Truncate(scaled);
                    break;
                default:
                    rounded = scaled;
                    break;
            }

            result.Value = rounded / multiplier;
        }
    }
}
