using UnityEngine;
using UnityEngine.Audio;
using HutongGames.PlayMaker;

namespace HutongGames.PlayMaker.Actions
{
    [ActionCategory("LV/Audio")]
    [HutongGames.PlayMaker.Tooltip("複数のAudioMixer音量設定を一括管理・保存・ロードします。")]
    public class LvMixerVolumeManager : FsmStateAction
    {
        [RequiredField]
        [HutongGames.PlayMaker.Tooltip("操作対象のAudioMixerアセット")]
        public AudioMixer audioMixer;

        [ActionSection("Volume Lists")]
        [RequiredField]
        [HutongGames.PlayMaker.Tooltip("ミキサー側でExposeした名前のリスト")]
        public FsmString[] parameterNames;

        [RequiredField]
        [UIHint(UIHint.Variable)]
        [HutongGames.PlayMaker.Tooltip("スライダーと連動させる変数のリスト (0.0 - 1.0)")]
        public FsmFloat[] volumeVariables;

        [HutongGames.PlayMaker.Tooltip("PlayerPrefsで使用するセーブキーのリスト")]
        public FsmString[] saveKeys;

        [ActionSection("Persistence Settings")]
        [HutongGames.PlayMaker.Tooltip("起動時にセーブデータから値を読み込むならチェック")]
        public bool loadOnStart = true;

        [HutongGames.PlayMaker.Tooltip("値が変更されるたびに自動でPlayerPrefsに書き込むならチェック")]
        public bool autoSave = true;

        [ActionSection("Runtime Settings")]
        [HutongGames.PlayMaker.Tooltip("スライダー操作に合わせて毎フレーム実行")]
        public bool everyFrame = true;

        // 内部保持用の前回値（変更があった項目だけ処理するため）
        private float[] lastVolumes;

        public override void Reset()
        {
            audioMixer = null;
            parameterNames = new FsmString[0];
            volumeVariables = new FsmFloat[0];
            saveKeys = new FsmString[0];
            loadOnStart = true;
            autoSave = true;
            everyFrame = true;
        }

        public override void OnEnter()
        {
            if (audioMixer == null) 
            {
                LogError("Audio Mixer is not assigned!");
                return;
            }

            // 配列のサイズチェック（各リストの数が合っていない場合は警告）
            int count = volumeVariables.Length;
            lastVolumes = new float[count];

            for (int i = 0; i < count; i++)
            {
                // 1. ロード処理
                if (loadOnStart && i < saveKeys.Length && !string.IsNullOrEmpty(saveKeys[i].Value))
                {
                    if (PlayerPrefs.HasKey(saveKeys[i].Value))
                    {
                        volumeVariables[i].Value = PlayerPrefs.GetFloat(saveKeys[i].Value);
                    }
                }

                // 2. 初回反映
                ApplyVolume(i);
            }

            if (!everyFrame) Finish();
        }

        public override void OnUpdate()
        {
            for (int i = 0; i < volumeVariables.Length; i++)
            {
                // 値に変化があった項目（スライダーが動かされた項目）だけを更新
                if (volumeVariables[i].Value != lastVolumes[i])
                {
                    ApplyVolume(i);
                }
            }
        }

        void ApplyVolume(int index)
        {
            if (index >= parameterNames.Length || string.IsNullOrEmpty(parameterNames[index].Value)) return;

            // 0.0-1.0をデシベルに変換
            float vol = Mathf.Clamp(volumeVariables[index].Value, 0.0001f, 1f);
            float dB = Mathf.Log10(vol) * 20f;

            // ミキサーに反映
            audioMixer.SetFloat(parameterNames[index].Value, dB);

            // セーブ処理
            if (autoSave && index < saveKeys.Length && !string.IsNullOrEmpty(saveKeys[index].Value))
            {
                PlayerPrefs.SetFloat(saveKeys[index].Value, volumeVariables[index].Value);
            }

            // 変更検知用キャッシュを更新
            lastVolumes[index] = volumeVariables[index].Value;
        }
    }
}