using UnityEngine;
using HutongGames.PlayMaker;

[ActionCategory("LV/Movement")]
[UnityEngine.Tooltip("Move an object from A to C through B with arc, scale, slow motion, and Euler rotation.")]
public class LvTripleArc : FsmStateAction
{
    [RequiredField]
    public FsmOwnerDefault targetObject;

    public FsmVector3 pointA;
    public FsmVector3 pointB;
    public FsmVector3 pointC;

    public FsmVector3 scaleA;
    public FsmVector3 scaleB;
    public FsmVector3 scaleC;

    public FsmVector3 rotationEulerA;
    public FsmVector3 rotationEulerB;
    public FsmVector3 rotationEulerC;

    public FsmFloat duration;
    public FsmFloat slowMotionFactor;
    public FsmFloat slowMotionDuration;

    private Transform target;
    private float time;
    private float slowStart;
    private float slowEnd;
    private bool isSlowing;

    public override void Reset()
    {
        duration = 1f;
        slowMotionFactor = 0.2f;
        slowMotionDuration = 0.5f;

        scaleA = new Vector3(1f, 1f, 1f);
        scaleB = new Vector3(1f, 1f, 1f);
        scaleC = new Vector3(1f, 1f, 1f);

        rotationEulerA = new Vector3(0f, 0f, 0f);     // �������
        rotationEulerB = new Vector3(0f, 90f, 0f);    // ������
        rotationEulerC = new Vector3(0f, 180f, 0f);   // ��������
    }

    public override void OnEnter()
    {
        target = Fsm.GetOwnerDefaultTarget(targetObject).transform;
        time = 0f;

        float midTime = duration.Value / 2f;
        slowStart = midTime - (slowMotionDuration.Value / 2f);
        slowEnd = midTime + (slowMotionDuration.Value / 2f);
        isSlowing = false;
    }

    public override void OnUpdate()
    {
        float delta = Time.deltaTime;

        if (time >= slowStart && time <= slowEnd)
        {
            delta *= slowMotionFactor.Value;
            isSlowing = true;
        }
        else if (isSlowing)
        {
            isSlowing = false;
        }

        time += delta;
        float t = Mathf.Clamp01(time / duration.Value);

        // �ʒu���
        Vector3 pos = CalculateArcPosition(t);
        target.position = pos;

        // �X�P�[�����
        Vector3 scale = CalculateArcScale(t);
        target.localScale = scale;

        // ��]��ԁiEuler�p�j
        Quaternion rot = CalculateArcRotation(t);
        target.rotation = rot;

        if (t >= 1f)
        {
            Finish();
        }
    }

    private Vector3 CalculateArcPosition(float t)
    {
        Vector3 ab = Vector3.Lerp(pointA.Value, pointB.Value, t);
        Vector3 bc = Vector3.Lerp(pointB.Value, pointC.Value, t);
        return Vector3.Lerp(ab, bc, t);
    }

    private Vector3 CalculateArcScale(float t)
    {
        Vector3 ab = Vector3.Lerp(scaleA.Value, scaleB.Value, t);
        Vector3 bc = Vector3.Lerp(scaleB.Value, scaleC.Value, t);
        return Vector3.Lerp(ab, bc, t);
    }

    private Quaternion CalculateArcRotation(float t)
    {
        Vector3 ab = Vector3.Lerp(rotationEulerA.Value, rotationEulerB.Value, t);
        Vector3 bc = Vector3.Lerp(rotationEulerB.Value, rotationEulerC.Value, t);
        Vector3 finalEuler = Vector3.Lerp(ab, bc, t);
        return Quaternion.Euler(finalEuler);
    }
}