using UnityEngine;
using HutongGames.PlayMaker;

[ActionCategory("LV/Movement")]
[HutongGames.PlayMaker.Tooltip("GameObjectを方向入力ベクトルに基づいて、単位（ブロック）数だけ滑らかに移動させます。斜め移動や補正の有無も選べます。")]
public class MoveGameObject_Grid2 : FsmStateAction
{
    [RequiredField]
    [HutongGames.PlayMaker.Tooltip("移動させるGameObject。")]
    public FsmOwnerDefault gameObject;

    [RequiredField]
    [HutongGames.PlayMaker.Tooltip("方向入力ベクトル。各軸の値が閾値を超えると、その軸方向に移動します。複数軸の同時入力も可能です。")]
    public FsmVector3 moveDirection;

    [HutongGames.PlayMaker.Tooltip("方向入力として有効とみなす最小値（絶対値）。")]
    public FsmFloat directionThreshold;

    [RequiredField]
    [HutongGames.PlayMaker.Tooltip("移動させる単位（ブロック）数。正の値を入力してください。")]
    public FsmFloat moveUnits;

    [RequiredField]
    [HutongGames.PlayMaker.Tooltip("移動にかかる時間（秒）。")]
    public FsmFloat moveDuration;

    [HutongGames.PlayMaker.Tooltip("移動完了時に送信するイベント。")]
    public FsmEvent finishEvent;

    [HutongGames.PlayMaker.Tooltip("移動をワールド座標で行うか、ローカル座標で行うか。")]
    public FsmBool isLocal;

    [HutongGames.PlayMaker.Tooltip("方向ベクトルを正規化して距離補正を行うかどうか。オフにするとグリッド単位で移動します。")]
    public FsmBool useNormalizedDirection;

    private float startTime;
    private Vector3 startPos;
    private Vector3 endPos;
    private GameObject go;
    private bool isFinished;

    public override void Reset()
    {
        gameObject = null;
        moveDirection = null;
        directionThreshold = 0.5f;
        moveUnits = 1.0f;
        moveDuration = 0.2f;
        finishEvent = null;
        isLocal = false;
        useNormalizedDirection = true;
    }

    public override void OnEnter()
    {
        go = Fsm.GetOwnerDefaultTarget(gameObject);

        if (go == null)
        {
            Finish();
            return;
        }

        isFinished = false;
        startPos = isLocal.Value ? go.transform.localPosition : go.transform.position;

        Vector3 dir = moveDirection.Value;
        float threshold = Mathf.Abs(directionThreshold.Value);
        Vector3 moveVec = Vector3.zero;

        if (Mathf.Abs(dir.x) > threshold) moveVec.x = Mathf.Sign(dir.x);
        if (Mathf.Abs(dir.y) > threshold) moveVec.y = Mathf.Sign(dir.y);
        if (Mathf.Abs(dir.z) > threshold) moveVec.z = Mathf.Sign(dir.z);

        if (moveVec == Vector3.zero)
        {
            Finish(); // 有効な入力がない場合は終了
            return;
        }

        Vector3 finalVec = useNormalizedDirection.Value
            ? moveVec.normalized * moveUnits.Value
            : moveVec * moveUnits.Value;

        endPos = startPos + finalVec;
        startTime = FsmTime.RealtimeSinceStartup;

        if (moveDuration.Value <= 0f)
        {
            ApplyPosition(endPos);
            Fsm.Event(finishEvent);
            Finish();
        }
    }

    public override void OnUpdate()
    {
        if (isFinished) return;

        float elapsedTime = FsmTime.RealtimeSinceStartup - startTime;
        float t = elapsedTime / moveDuration.Value;

        if (t < 1f)
        {
            Vector3 currentPos = Vector3.Lerp(startPos, endPos, t);
            ApplyPosition(currentPos);
        }
        else
        {
            ApplyPosition(endPos);
            isFinished = true;
            Fsm.Event(finishEvent);
            Finish();
        }
    }

    private void ApplyPosition(Vector3 position)
    {
        if (isLocal.Value)
        {
            go.transform.localPosition = position;
        }
        else
        {
            go.transform.position = position;
        }
    }
}
